//
//  WXKBookmarkWindow_DragAndDrop.m
//  AnotherKyoponUtilities
//
//  Created by FUJIDANA on 06/08/27.
//  Copyright 2006 FUJIDANA. All rights reserved.
//
//
//  Some part of this source code is derived from DNDArrayController.h, 
//  written by raktajino in AH-K3001V Bookmark Utility and distributed
//  under BSD license.
//  See the following about copyrights.
//
//
//  Created by raktajino on Thu Jun 17 2004.
//  Copyright (c) 2004 raktajino. All rights reserved.
//
//		Copyright (c) 04 Zoroyoshi, Japan
//		All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


#import "WXKBookmarkWindowController.h"
#import "FJDIndexedItemsArrayController.h"


static NSString *kWebURLsWithTitlesPboardType = @"WebURLsWithTitlesPboardType";


@interface WXKBookmarkWindowController (DragAndDrop_Private)

- (NSArray *)bookmarksFromPasteboard:(NSPasteboard *)pboard;
- (NSArray *)bookmarksFromWebURLsWithTitlesPasteboard:(NSPasteboard *)pboard;
- (NSManagedObject *)bookmarkFromURLPasteboard:(NSPasteboard *)pboard;


@end


#pragma mark -


@implementation WXKBookmarkWindowController (DragAndDrop)

- (NSArray *)readablePasteboardTypes
{
	return [[super readablePasteboardTypes] arrayByAddingObject:NSURLPboardType];
}

- (BOOL)tableView:(NSTableView *)tableView writeRowsWithIndexes:(NSIndexSet *)rowIndexes toPasteboard:(NSPasteboard *)pboard
{
	BOOL isWrittenBySuperClass = [super tableView:tableView
							 writeRowsWithIndexes:rowIndexes
									 toPasteboard:pboard];
	
	BOOL isWrittenByThisClass = NO;
	
	if (tableView == itemsTableView) {
		NSArray *selectedObjects = [[itemsController arrangedObjects] objectsAtIndexes:rowIndexes];
		
		// Write a URL into the paste board as NSURLPboardType
		NSManagedObject *object = [selectedObjects lastObject];
		NSString *URLString = [object valueForKey:@"url"];
		if (URLString && [URLString length] > 0) {
			NSURL *URL = [NSURL URLWithString:URLString];
			if (URL) {
				[pboard addTypes:[NSArray arrayWithObject:NSURLPboardType] owner:nil];
				[URL writeToPasteboard:pboard];
				isWrittenByThisClass = YES;
			}
		}
		
		// writing array of URLs into the paste board as NSURLPboardType
		NSArray *URLStrings = [selectedObjects valueForKey:@"url"];
		NSArray *titles = [selectedObjects valueForKey:@"title"];
		if (URLStrings && titles && [URLStrings count] != 0 && [titles count] != 0) {
			NSArray *webURLsWithTitles = [NSArray arrayWithObjects:URLStrings, titles, nil];
			[pboard addTypes:[NSArray arrayWithObject:kWebURLsWithTitlesPboardType] owner:nil];
			[pboard setPropertyList:webURLsWithTitles forType:kWebURLsWithTitlesPboardType];
			isWrittenByThisClass = YES;
		}
	}
	
	return isWrittenBySuperClass | isWrittenByThisClass;
}

- (NSDragOperation)tableView:(NSTableView *)tableView validateDrop:(id <NSDraggingInfo>)info proposedRow:(int)row proposedDropOperation:(NSTableViewDropOperation)operation
{
	NSDragOperation flag = [super tableView:tableView
							   validateDrop:info
								proposedRow:row
					  proposedDropOperation:operation];
	
	if (flag != NSDragOperationNone) {
		return flag;
	}
	
	if (tableView == itemsTableView
		&& operation == NSTableViewDropAbove 
		&& [info draggingSource] == nil // drop from other application
		&& [itemsController canInsert] // if multiple folder is selected, return NO (since itemsController can not decide in which folder new object should be inserted)
		&& [[info draggingPasteboard] availableTypeFromArray:[NSArray arrayWithObjects:kWebURLsWithTitlesPboardType, NSURLPboardType, nil]] != nil 		) {
		return NSDragOperationCopy;
	}
	
	if (tableView == foldersTableView
		&& operation == NSTableViewDropOn
		&& [info draggingSource] == nil // drop from other application
		&& [itemsController canAdd] // if multiple folder is selected, return NO (since itemsController can not decide in which folder new object should be inserted)
		&& [[info draggingPasteboard] availableTypeFromArray:[NSArray arrayWithObjects:kWebURLsWithTitlesPboardType, NSURLPboardType, nil]] != nil
		) {
		return NSDragOperationCopy;
	}
	
	return NSDragOperationNone;
}

- (BOOL)tableView:(NSTableView *)tableView acceptDrop:(id <NSDraggingInfo>)info row:(int)row dropOperation:(NSTableViewDropOperation)operation
{
	if ([super tableView:tableView
				  acceptDrop:info
						 row:row
			   dropOperation:operation]) {
		
		return YES;
	}
	
	if (tableView == itemsTableView
		&& operation == NSTableViewDropAbove 
		&& [info draggingSource] == nil // drop from other application
		&& [itemsController canInsert] // if multiple folder is selected, return NO (since itemsController can not decide in which folder new object should be inserted)
		&& [[info draggingPasteboard] availableTypeFromArray:[NSArray arrayWithObjects:kWebURLsWithTitlesPboardType, NSURLPboardType, nil]] != nil
		) {
			
		NSArray *bookmarks = [self bookmarksFromPasteboard:[info draggingPasteboard]];
		if (bookmarks && [bookmarks count] > 0) {
			NSIndexSet *indexes = [NSIndexSet indexSetWithIndexesInRange:NSMakeRange(row, [bookmarks count])];
			[itemsController insertObjects:bookmarks atArrangedObjectIndexes:indexes];
			return YES;
		}
	}
	
	if (tableView == foldersTableView
		&& operation == NSTableViewDropOn
		&& [info draggingSource] == nil // drop from other application
		&& [itemsController canAdd] // if multiple folder is selected, return NO (since itemsController can not decide in which folder new object should be inserted)
		&& [[info draggingPasteboard] availableTypeFromArray:[NSArray arrayWithObjects:kWebURLsWithTitlesPboardType, NSURLPboardType, nil]] != nil
		) {
		
		[foldersController setSelectionIndex:row];
		
		NSArray *bookmarks = [self bookmarksFromPasteboard:[info draggingPasteboard]];
		if (bookmarks && [bookmarks count] > 0) {
			[itemsController addObjects:bookmarks];
			return YES;
		}
	}
	return NO;
}

@end


#pragma mark -


@implementation WXKBookmarkWindowController (DragAndDrop_Private)

- (NSArray *)bookmarksFromPasteboard:(NSPasteboard *)pboard
{
	NSString *availablePboardType = [pboard availableTypeFromArray:[NSArray arrayWithObjects:kWebURLsWithTitlesPboardType, NSURLPboardType, nil]];
	
	if ([availablePboardType isEqualToString:kWebURLsWithTitlesPboardType]) {
		return [self bookmarksFromWebURLsWithTitlesPasteboard:pboard];
	}
	
	if ([availablePboardType isEqualToString:NSURLPboardType]) {
		NSManagedObject *bookmark = [self bookmarkFromURLPasteboard:pboard];
		if (bookmark) return [NSArray arrayWithObject:bookmark];
	}
	
	return nil;
}

- (NSArray *)bookmarksFromWebURLsWithTitlesPasteboard:(NSPasteboard *)pboard
{
	NSMutableArray *bookmarks;
	
	NSArray *array = [pboard propertyListForType:kWebURLsWithTitlesPboardType];
	if (array == nil || [array count] < 2) return nil;
	
	NSArray *URLStrings = [array objectAtIndex:0];
	NSArray *titles = [array objectAtIndex:1];
	
	bookmarks = [NSMutableArray arrayWithCapacity:[URLStrings count]];
	
	NSEnumerator *URLStringEnumerator = [URLStrings objectEnumerator];
	NSEnumerator *titleEnumerator = [titles objectEnumerator];
	NSString *URLString, *title;
	
	while ((URLString = [URLStringEnumerator nextObject]) && (title = [titleEnumerator nextObject])) {
		NSManagedObject *bookmark = [NSEntityDescription insertNewObjectForEntityForName:@"Bookmark"
																  inManagedObjectContext:[self managedObjectContext]];
		[bookmark setValue:URLString forKey:@"url"];
		[bookmark setValue:title forKey:@"title"];
		[bookmarks addObject:bookmark];
	}
	return bookmarks;
}

- (NSManagedObject *)bookmarkFromURLPasteboard:(NSPasteboard *)pboard
{
	NSURL *URL = [NSURL URLFromPasteboard:pboard];
	if (URL == nil) return nil;
	
	// If an URL in the pasteboard indicates an internet location file, extract its title and web URL.
	
	NSString *URLStringInResource = nil;
	NSString *titleInResource = nil;
	FSRef  bookmarkFileRef;
	
	if (CFURLGetFSRef((CFURLRef)URL, &bookmarkFileRef)) {
		short refNum = FSOpenResFile(&bookmarkFileRef, fsCurPerm);
		if (refNum != -1) {
			Handle hURL   = Get1IndResource('url ', 1);
			Handle hTitle = Get1IndResource('urln', 1);
			if (hURL) {
				HLock(hURL);
				URLStringInResource = [[[NSString alloc] initWithBytes:*hURL length:GetHandleSize(hURL) encoding:NSUTF8StringEncoding] autorelease];
				HUnlock(hURL);
			}
			if (hTitle) {
				HLock(hTitle);
				titleInResource = [[[NSString alloc] initWithBytes:*hTitle length:GetHandleSize(hTitle) encoding:NSUTF8StringEncoding] autorelease];
				HUnlock(hTitle);
			}
			CloseResFile(refNum);
		}
	}
	
	NSManagedObject *bookmark = [NSEntityDescription insertNewObjectForEntityForName:@"Bookmark"
															  inManagedObjectContext:[self managedObjectContext]];
	
	if (URLStringInResource && titleInResource) {
		// Since a title and an web URL has been successfully extracted, set them as properties of a new bookmark.
		[bookmark setValue:URLStringInResource forKey:@"url"];
		[bookmark setValue:titleInResource forKey:@"title"];
		
	} else {
		// Failed to extract a title or an web URL.
		// The URL in the pasteboard may not indicate an internet location file, so simply set the URL as a property of a new bookmark.
		[bookmark setValue:[URL absoluteString] forKey:@"url"];
	}
	
	return bookmark;
}

@end
